pro initialize

common paths, work_dir
common times, timall, start_date, end_date, syr, sdoy, eyr, edoy
common files, input_file, save_file, output_file, output_file2, create_save_file

; Variable       Sample Value                    Default Value      Meaning
; -----------    --------------                  -----------------  ------------------------------------
; work_dir       $WORK:[BLEAU]                   $WORK:[PRODUCTION] location of work files, data files, etc.
; start_date     2004_029                        none               starting date of data set
; end_date       2004_033                        none               ending date of data set
; input_file1    PHASPECT_2004_029_2004_033.PHA  none               name of spect input file
; input_file2    PHSMOOSH_2004_029_2004_033.PHA  none               name of smoosh input file
; input_file3    PHAMASS_2004_029_2004_033.PHA   none               name of mass input file
; output_file    ULEIS_2004_029_2004_033.PS      input_file1        name of output PS file
; output_file2   ULEIS_2004_029_2004_033.PNG     input_file1        name of output PNG file
; create_save_file  y                            y                  boolean flag, whether or
;                                                                   not to create a .sav file
; 
; Notes: The three input files would be located in work_dir by default, but may be
; located elsewhere is so desired; they would be complete filenames.  For the
; output files, IDL code would append '.ps' or '.png' to output_file depending on
; the output format and would write the file into work_dir.  If output_file or
; output_file2 isn't given we will use the filename portion of input_file1.

; Obtain value of DCL symbol WORK_DIR
; This is where to look for .PHA files if no directory is specified
; Defaults to $WORK:[PRODUCTION]
;
; Note: If logical name WORK_DIR is also defined, this will obtain the
; logical name's value, *not* the DCL symbol's value.

work_dir = getenv("work_dir")
if work_dir eq "" then work_dir = "$WORK:[PRODUCTION]"
print
print,"Using work_dir = ",work_dir

; Obtain value of DCL symbol START_DATE
; This is the date, in the format yyyy_ddd, of the start of the plot
; This symbol is required

start_date = getenv("start_date")
if start_date eq "" or strlen(start_date) ne 8 or strmid(start_date,4,1) ne '_' then begin
  print,"DCL symbol START_DATE must be defined, format yyyy_nnn"
  stop
endif

; Obtain value of DCL symbol END_DATE
; This is the date, in the format yyyy_ddd, of the day at the end of the plot,
; *not* the last full day of the plot.
; This symbol is required

end_date = getenv("end_date")
if end_date eq "" or strlen(end_date) ne 8 or strmid(end_date,4,1) ne '_' then begin
  print,"DCL symbol END_DATE must be defined, format yyyy_nnn"
  stop
endif

; Define integer array to hold start and end dates/times as follows:
;
;   timarr(0)   start year
;   timarr(1)   start day of year
;   timarr(2)   start hour
;   timarr(3)   start minute
;   timarr(4)   end year
;   timarr(5)   end day of year
;   timarr(6)   end hour
;   timarr(7)   end minute

timall = intarr(8)

; Decode the values of the strings in start_date and end_date and fill in
; timarr(0), timarr(1), timarr(4), and timarr(5).
; Assume we always start and end plots on day boundaries, so leave other
; elements of timarr zero.

reads, start_date, format='(i4,1x,i3)', syr, sdoy
timall(0) = syr
timall(1) = sdoy

reads, end_date, format='(i4,1x,i3)', eyr, edoy
timall(4) = eyr
timall(5) = edoy

print,'Start date =',start_date
print,'Stop date  =',end_date
print,timall,format="('timall = ',8i5)"

; Obtain values for DCL symbols input_file1, input_file2, and input_file3
; These are the names of the input pha data files for the species, smoosh, and
; mass data, respectively.
; These sybmols are required
; Store them into the string array input_file

input_file = strarr(3)

for i=0,2 do begin
  symbolname = string(i+1,format="('input_file',i1)")
  input_file(i) = getenv(symbolname)
  if input_file(i) eq "" then begin
    print,"DCL symbol ",symbolname," must be defined as name of input file",i+1
    stop
  endif
endfor

; For each input_file value, if it does not contain either a device or a
; directory name (ending with ':' or ']', respectively), prefix the value
; of the variable work_dir.

for i=0,2 do begin
  if strpos(input_file(i),']') eq -1 or strpos(input_file(i),':') eq -1 then $
    input_file(i) = work_dir+input_file(i)
  print,'input_file(',i,') = ',input_file(i) 
endfor

; Obtain value for DCL symbol CREATE_SAVE_FILE
; This is a one-character flag equal to either 'y', 'Y', or '1' if a save file
; is to be created, anything else if it is not to be created.
; The default value is 'y' (create save file).
; Save flag as integer 1 or 0 in create_save_file.

str_create_save_file = getenv("create_save_file")
if str_create_save_file eq "" then str_create_save_file = 'y'
if (str_create_save_file eq 'Y') or $
   (str_create_save_file eq 'y') or $
   (str_create_save_file eq '1') then begin
  create_save_file = 1
endif else begin
  create_save_file = 0
endelse

; Determine names of save files, store into string array save_file
; Derive these from the corresponding values of input_file by changing the
; file type to ".SAV".

save_file = strarr(3)

for i=0,2 do begin
  save_file(i) = input_file(i)
  pos = strpos(save_file(i),']') > strpos(save_file(i),':')
  pos2 = strpos(input_file(i),'.',pos+1)
  if pos2 ge 0 then begin
    save_file(i) = strmid(save_file(i),0,pos2) + '.SAV'
  endif else begin
    save_file(i) = save_file(i) + '.SAV'
  endelse
print,'save_file(',i,') = ',save_file(i) 
endfor

; Obtain value of DCL symbol OUTPUT_FILE
; This is the name of the PS plot file that will eventually be written
; If not defined, derive it from save_file(0) as follows:
;   Locate part of name portion of save_file(0) up to the first '_'
;   Replace this part of the name with 'ULEIS_'
;   Locate file type, replace it with '.PS'; if no file type append '.PS'
;   Preserve device and directory; if not present use value of work_dir

output_file = getenv("output_file")
if output_file eq "" then begin
  pos = strpos(save_file(0),']') > strpos(save_file(0),':')
  pos2 = strpos(save_file(0),'_',pos+1)
  pos3 = strpos(save_file(0),'.',pos+1)
  if pos3 ge 0 then begin
;print,"'",strmid(save_file(0),pos2+1,pos3-(pos2+1)),"'"
    output_file = 'ULEIS_' + strmid(save_file(0),pos2+1,pos3-(pos2+1)) + '.PS'
  endif else begin
;print,"'",strmid(save_file(0),pos2+1,strlen(save_file(0))-(pos2+1)),"'"
    output_file = 'ULEIS_' + strmid(save_file(0),pos2+1,strlen(save_file(0))-(pos2+1)) + '.PS'
  endelse
  if pos ge 0 then $
    output_file = strmid(save_file(0),0,pos+1) + output_file $
  else $
    output_file = work_dir + output_file
endif

; Obtain value of DCL symbol OUTPUT_FILE2
; This is the name of the PNG plot file that will eventually be written
; If not defined, derive it from save_file(0) as follows:
;   Locate part of name portion of save_file(0) up to the first '_'
;   Replace this part of the name with 'ULEIS_'
;   Locate file type, replace it with '.PNG'; if no file type append '.PNG'
;   Preserve device and directory; if not present use value of work_dir

output_file2 = getenv("output_file2")
if output_file2 eq "" then begin
  pos = strpos(save_file(0),']') > strpos(save_file(0),':')
  pos2 = strpos(save_file(0),'_',pos+1)
  pos3 = strpos(save_file(0),'.',pos+1)
  if pos3 ge 0 then begin
;print,"'",strmid(save_file(0),pos2+1,pos3-(pos2+1)),"'"
    output_file2 = 'ULEIS_' + strmid(save_file(0),pos2+1,pos3-(pos2+1)) + '.PNG'
  endif else begin
;print,"'",strmid(save_file(0),pos2+1,strlen(save_file(0))-(pos2+1)),"'"
    output_file2 = 'ULEIS_' + strmid(save_file(0),pos2+1,strlen(save_file(0))-(pos2+1)) + '.PNG'
  endelse
  if pos ge 0 then $
    output_file2 = strmid(save_file(0),0,pos+1) + output_file2 $
  else $
    output_file2 = work_dir + output_file2
endif

print,'Will write ps plot to ',output_file
print,'Will write png plot to ',output_file2

end
